/*

pitchEnv~

Copyright 2011 William Brent

This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
You should have received a copy of the GNU General Public License along with this program.  If not, see <http://www.gnu.org/licenses/>.

version 0.2, February 24, 2011

 0.2 adds the denoising feature originally done as a separate extern in purePitch~

*/

#define NUMHARMONICS 40
#include "m_pd.h"
#include <math.h>
#ifndef M_PI
#define M_PI 3.1415926535897932384626433832795
#endif

static t_class *pitchEnv_tilde_class;

typedef struct _pitchEnv_tilde 
{
    t_object x_obj;
    t_symbol *x_arrayname;
	int x_array_points;
	int arrayFlag;
	int denoise;
	t_word *x_vec;
    t_float sr;
    t_float n;
    int dsp_tick;
    int buffer_limit;
    int overlap;
    int overlap_chunk;
	int pad;
    int window;
    int window_half;
	t_float ffa;
    t_float amp_scalar;
    t_float fundamental;
	t_float harmonics[NUMHARMONICS];
	int harmonicsBins[NUMHARMONICS];
    t_float env[NUMHARMONICS];
    t_float *hann;
	t_sample *signal_buf;
	t_sample *signal_buf_windowed;
	t_float *signal_R;
	t_float *signal_I;
	
	t_sample *nonoverlapped_output;
	t_sample *final_output;
	
    t_float x_f;
    
} pitchEnv_tilde;


/* ------------------------ pitchEnv~ -------------------------------- */

static void pitchEnv_tilde_set(pitchEnv_tilde *x, t_symbol *s)
{
	t_garray *a;
	int i;

	if(!(a = (t_garray *)pd_findbyclass(s, garray_class)))
	{
		pd_error(x, "%s: no such array", s->s_name);
		x->arrayFlag = 0;
	}
	else if(!garray_getfloatwords(a, &x->x_array_points, &x->x_vec))
	{
		pd_error(x, "%s: bad template for pitchEnv~", s->s_name);
		x->arrayFlag = 0;
	}
	else
	{
	    x->x_arrayname = s;

		// get the current table values
		for(i=0; i<x->x_array_points && i<NUMHARMONICS; i++)
		{
			x->env[i] = x->x_vec[i].w_float;
			x->env[i] = (x->env[i]<0 ? 0 : x->env[i]);
		}
		
		// if no scalar is specified for higher harmonics, flag with -1
		for(; i<NUMHARMONICS; i++)
			x->env[i] = -1;
		
		x->arrayFlag = 1;
	}
}		
		
static void pitchEnv_tilde_amp_scalar(pitchEnv_tilde *x, t_floatarg a)
{
	x->amp_scalar = a/x->window;
	
	post("amp scalar: %f", x->amp_scalar);
}

static void pitchEnv_tilde_pad(pitchEnv_tilde *x, t_floatarg p)
{
	x->pad = (p<0 ? 0 : p);
	x->pad = (x->pad>x->window_half ? x->window_half : x->pad);
	
	post("bin padding: %i", x->pad);
}

static void pitchEnv_tilde_denoise(pitchEnv_tilde *x, t_floatarg dn)
{
	x->denoise = (dn<0 ? 0 : dn);
	x->denoise = (x->denoise>1 ? 1 : x->denoise);
	
	if(x->denoise)
		post("denoise ON");
	else
		post("denoise OFF");
}

static void pitchEnv_tilde_fundamental(pitchEnv_tilde *x, t_floatarg fund)
{
    if (fund <= -1500)
    	x->fundamental = 1;
    else if (fund > 130)
    	x->fundamental = (8.17579891564 * exp(.0577622650 * 130));
    else
    	x->fundamental = (8.17579891564 * exp(0.0577622650 * fund));
}

static void *pitchEnv_tilde_new(t_symbol *s, t_floatarg window, t_floatarg overlap)
{
    pitchEnv_tilde *x = (pitchEnv_tilde *)pd_new(pitchEnv_tilde_class);
	int i, isPow2;
	
    inlet_new(&x->x_obj, &x->x_obj.ob_pd, gensym("float"), gensym("fundamental"));
	outlet_new(&x->x_obj, gensym("signal"));

	if(window)
	{
		isPow2 = (int)window && !( ((int)window-1) & (int)window );
		if(isPow2)
			x->window = window;
		else
		{
			error("pitchEnv~: window size must be a power of 2.");
			x->window = 4096;
		}
	}
	else
		x->window = 4096;


	if(overlap)
	{
		isPow2 = (int)overlap && !( ((int)overlap-1) & (int)overlap );
		if(isPow2)
			x->overlap = overlap;
		else
		{
			error("pitchEnv~: overlap must be a power of 2.");
			x->overlap = 4;
		}
	}
	else
		x->overlap = 4;
		
	
	x->sr = 44100;
	x->n = 64;
	x->dsp_tick = 0; 
	x->window_half = x->window/2.0;
	x->overlap_chunk = x->window/x->overlap;
	x->amp_scalar = (2.0/3.0)/x->window;
	x->fundamental = 440.0;
	x->pad = floor(x->window * 0.00052);
	x->ffa = x->sr/x->window;
	x->x_arrayname = s;
	x->arrayFlag = 0;
	x->denoise = 0;
	
	x->buffer_limit = x->window/x->n/x->overlap; // divide by overlap since we want to push out buffered audio to a window when the main buffer has been updated by 1/overlap the window sizse.
	
	x->hann = (t_float *)getbytes(x->window * sizeof(t_float));
	x->signal_buf = (t_sample *)getbytes(x->window * sizeof(t_sample));
	x->signal_buf_windowed = (t_sample *)getbytes(x->window * sizeof(t_sample));
	x->signal_R = (t_float *)getbytes((x->window_half+1) * sizeof(t_float));
	x->signal_I = (t_float *)getbytes((x->window_half+1) * sizeof(t_float));
	x->nonoverlapped_output = (t_sample *)getbytes((x->window*x->overlap) * sizeof(t_sample));
	x->final_output = (t_sample *)getbytes((x->overlap_chunk) * sizeof(t_sample));

	// initialize env faders
 	for(i=0; i<NUMHARMONICS; i++)
 		x->env[i] = -1;
 		
	// initialize hann window
 	for(i=0; i<x->window; i++)
 		x->hann[i] = 0.5 * (1 - cos(2*M_PI*i/x->window));

 	for(i=0; i<x->window; i++)
 	{
 		x->signal_buf[i] = 0.0;
 		x->signal_buf_windowed[i] = 0.0;
	};

	// init output buffer
	for(i=0; i<(x->overlap_chunk); i++)
		x->final_output[i] = 0.0;
 			
	// initialize nonoverlapped_output
 	for(i=0; i<(x->window*x->overlap); i++)
 		x->nonoverlapped_output[i] = 0.0;
	
	// init (window_half+1 size arrays)
	for(i=0; i<=x->window_half; i++)
	{
		x->signal_R[i] = 0.0;
		x->signal_I[i] = 0.0;
	};	
	
    post("pitchEnv~: window size: %i, overlap: %i, bin padding: %i", x->window, x->overlap, x->pad);
    
    return (x);
}


static t_int *pitchEnv_tilde_perform(t_int *w)
{
    int i, j, n, window, window_half, overlap, overlap_chunk;
    t_float amp_scalar;

    pitchEnv_tilde *x = (pitchEnv_tilde *)(w[1]);

    t_sample *in = (t_float *)(w[2]);
    t_sample *out = (t_float *)(w[3]);
    
    if(!x->arrayFlag)
    	pitchEnv_tilde_set(x, x->x_arrayname);
		
    n = w[4];
	window = x->window;
	window_half = x->window_half;
	overlap = x->overlap;
	overlap_chunk = x->overlap_chunk;
	
	amp_scalar = x->amp_scalar;
	
	// shift previous contents back
	for(i=0; i<(window-n); i++)
		x->signal_buf[i] = x->signal_buf[n+i];
		
	// buffer most recent block
	for(i=0; i<n; i++)
		x->signal_buf[window-n+i] = in[i];
		
	
	if(x->dsp_tick>=x->buffer_limit)
	{
		int prevRangeHi;
		
		prevRangeHi = 0;
		x->dsp_tick = 0;
		
		// window the signal
		for(i=0; i<window; i++)
			x->signal_buf_windowed[i] = x->signal_buf[i] * x->hann[i];
	
		// take FT of window
		mayer_realfft(window, x->signal_buf_windowed);
	
		// unpack mayer_realfft results into R&I arrays
		for(i=0; i<=window_half; i++)
		{
			x->signal_R[i] = x->signal_buf_windowed[i];
			if(fabs(x->signal_R[i]) < 0.0001)
				x->signal_R[i] = 0.0;
		}
		
		x->signal_I[0]=0;  // DC
		
		for(i=(window-1), j=1; i>window_half; i--, j++)
		{
			x->signal_I[j] = x->signal_buf_windowed[i];
			if(fabs(x->signal_I[j]) < 0.0001)
				x->signal_I[j] = 0.0;
		}
		
		x->signal_I[window_half]=0;  // Nyquist

		// find harmonics for the current fundamental
		for(i=0; i<NUMHARMONICS; i++)
		{
			x->harmonics[i] = x->fundamental * (i+1);
			
			if(x->harmonics[i] > x->sr*0.5)
				x->harmonicsBins[i] = -1; // if a harmonic goes over Nyquist, flag it
			else
				x->harmonicsBins[i] = floor(x->harmonics[i]/x->ffa + 0.5);
		}


		// get the current table values
		for(i=0; i<x->x_array_points && i<NUMHARMONICS; i++)
		{
			x->env[i] = x->x_vec[i].w_float;
			x->env[i] = (x->env[i]<0 ? 0 : x->env[i]);
		}
		
		// if no scalar is specified for higher harmonics, flag with -1
		for(; i<NUMHARMONICS; i++)
			x->env[i] = -1;
		
		for(i=0; i<NUMHARMONICS; i++)
		{
			int peak, rangeLo, rangeHi;
			t_float fader;
			
			// leave harmonics with unspecified scalars (flagged as -1s) alone
			fader = (x->env[i]<0 ? 1 : x->env[i]);
			peak = x->harmonicsBins[i];

			// if x->harmonicsBins[i] had the -1 flag, there's no need to keep processing harmonics, so break
			if(peak<0)
				break;
			
			rangeLo = peak - x->pad;
			rangeHi = peak + x->pad;

			rangeLo = (rangeLo<0 ? 0 : rangeLo);
			rangeHi = (rangeHi>window_half ? window_half : rangeHi);

			if(x->denoise)
			{
				//post("prh: %i, rl: %i, rh: %i", prevRangeHi, rangeLo, rangeHi);
				for(j=prevRangeHi; j<rangeLo; j++)
				{
					x->signal_R[j] = 0.0;
					x->signal_I[j] = 0.0;
				}
			}
			
			for(j=rangeLo; j<=rangeHi; j++)
			{
				x->signal_R[j] *= fader;
				x->signal_I[j] *= fader;
			}
			
			prevRangeHi = rangeHi+1;
		}

		if(x->denoise)
		{
			//post("prh: %i, wh: %i", prevRangeHi, window_half);
			for(i=prevRangeHi; i<=window_half; i++)
			{
				x->signal_R[i] = 0.0;
				x->signal_I[i] = 0.0;
			}
		}
		
		// pack real and imaginary parts in correct order for mayer_realifft
		for(i=0; i<=window_half; i++)
			x->signal_buf_windowed[i] = x->signal_R[i];
		
		for(i=(window_half+1), j=(window_half-1); i<window; i++, j--)
			x->signal_buf_windowed[i] = x->signal_I[j];
			
		// resynth
		mayer_realifft(window, x->signal_buf_windowed);
			
			
		// window
		for(i=0; i<window; i++)
		{
			x->signal_buf_windowed[i] *= x->hann[i];
			x->signal_buf_windowed[i] *= amp_scalar;
		}
			
		// shift nonoverlapped blocks back
		for(i=0; i<(overlap-1); i++)
			for(j=0; j<window; j++)
				x->nonoverlapped_output[(i*window)+j] = x->nonoverlapped_output[((i+1)*window)+j];
		
		// write the new block
		for(i=0; i<window; i++)
			x->nonoverlapped_output[((overlap-1)*window)+i] = x->signal_buf_windowed[i];
			
		// init this chunk of the final output so it can be summed in the for() below
		for(i=0; i<overlap_chunk; i++)
			x->final_output[i] = 0.0;
			
		// do the overlap/add
		for(i=0; i<overlap; i++)
			for(j=0; j<overlap_chunk; j++)
				x->final_output[j] += x->nonoverlapped_output[(i*window)+((overlap-i-1)*overlap_chunk)+j];
	};
	
	// output
	for(i=0; i<n; i++, out++)
		*out = x->final_output[(x->dsp_tick*n)+i];

	x->dsp_tick++;

    return (w+5);
}


static void pitchEnv_tilde_dsp(pitchEnv_tilde *x, t_signal **sp)
{
	dsp_add(
		pitchEnv_tilde_perform,
		4,
		x,
		sp[0]->s_vec,
		sp[1]->s_vec,
		sp[0]->s_n
	);

// compare n to stored n and recalculate hann & filterbank if different
	if( sp[0]->s_n != x->n  ||  sp[0]->s_sr != x->sr )
	{
		x->sr = sp[0]->s_sr;
		x->n = sp[0]->s_n;
		
		x->buffer_limit = x->window/x->n/x->overlap;
	};

};

static void pitchEnv_tilde_free(pitchEnv_tilde *x)
{	
    t_freebytes(x->hann, x->window*sizeof(t_float));
    t_freebytes(x->signal_buf, x->window*sizeof(t_sample));
    t_freebytes(x->signal_buf_windowed, x->window*sizeof(t_sample));
    t_freebytes(x->signal_R, (x->window_half+1)*sizeof(t_float));
    t_freebytes(x->signal_I, (x->window_half+1)*sizeof(t_float));
    t_freebytes(x->nonoverlapped_output, (x->window*x->overlap)*sizeof(t_sample));
    t_freebytes(x->final_output, (x->overlap_chunk)*sizeof(t_sample));
};

void pitchEnv_tilde_setup(void)
{
    pitchEnv_tilde_class = 
    class_new(
    	gensym("pitchEnv~"),
    	(t_newmethod)pitchEnv_tilde_new,
    	(t_method)pitchEnv_tilde_free,
        sizeof(pitchEnv_tilde),
        CLASS_DEFAULT,
        A_DEFSYMBOL,
        A_DEFFLOAT,
        A_DEFFLOAT,
		0
    );

    CLASS_MAINSIGNALIN(pitchEnv_tilde_class, pitchEnv_tilde, x_f);

	class_addmethod(
		pitchEnv_tilde_class, 
        (t_method)pitchEnv_tilde_set,
		gensym("set"),
		A_DEFSYMBOL,
		0
	);
	
	class_addmethod(
		pitchEnv_tilde_class, 
        (t_method)pitchEnv_tilde_amp_scalar,
		gensym("amp_scalar"),
		A_DEFFLOAT,
		0
	);

	class_addmethod(
		pitchEnv_tilde_class, 
        (t_method)pitchEnv_tilde_denoise,
		gensym("denoise"),
		A_DEFFLOAT,
		0
	);
	
	class_addmethod(
		pitchEnv_tilde_class, 
        (t_method)pitchEnv_tilde_pad,
		gensym("pad"),
		A_DEFFLOAT,
		0
	);
	
	class_addmethod(
		pitchEnv_tilde_class, 
        (t_method)pitchEnv_tilde_fundamental,
		gensym("fundamental"),
		A_DEFFLOAT,
		0
	);
	
    class_addmethod(
    	pitchEnv_tilde_class,
    	(t_method)pitchEnv_tilde_dsp,
    	gensym("dsp"),
    	0
    );
}
